 
import Foundation

// MARK: - 1. SafeURL 包装器
struct SafeURL: Codable {
  var value: URL?

  init(from decoder: Decoder) throws {
    let container = try decoder.singleValueContainer()
    let urlString = try container.decode(String.self)
    value = URL(string: urlString)
  }

  func encode(to encoder: Encoder) throws {
    var container = encoder.singleValueContainer()
    try container.encode(value?.absoluteString ?? "")
  }
}

// MARK: - 2. 枚举容错
enum Status: String, Codable {
  case todo, doing, done, unknown

  init(from decoder: Decoder) throws {
    let container = try decoder.singleValueContainer()
    let raw = try container.decode(String.self)
    self = Status(rawValue: raw) ?? .unknown
  }
}

// MARK: - 3. 子结构体
struct Organizer: Codable {
  var name: String
  var url: SafeURL
}

// MARK: - 4. 主结构体
struct Event: Codable {
  var title: String
  var status: Status
  var date: Date
  var organizer: Organizer
}

// MARK: - 5. JSON 示例
let json = """
{
  "title": "Swift Workshop",
  "status": "archived",
  "date": "2025-09-27 14:30:00",
  "organizer": {
    "name": "Ccapton",
    "url": "not_a_valid_url"
  }
}
""".data(using: .utf8)!

// MARK: - 6. JSONDecoder 配置
let decoder = JSONDecoder()
let formatter = DateFormatter()
formatter.dateFormat = "yyyy-MM-dd HH:mm:ss"
decoder.dateDecodingStrategy = .formatted(formatter)

// MARK: - 7. 尝试解码
do {
  let event = try decoder.decode(Event.self, from: json)
  print("标题：\(event.title)")
  print("状态：\(event.status)")
  print("日期：\(event.date)")
  print("组织者：\(event.organizer.name)")
  print("组织者URL：\(event.organizer.url.value?.absoluteString ?? "无效URL")")
} catch {
  print("解码失败：\(error)")
}
